package gc;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
public class GCDemo {
    public static final int BYTES_IN_MB = 1024 * 1024;
    public static final int DEFAULT_BLOCK_SIZE = 10;
    public static final int DEFAULT_COUNT = 2;

    private int blockSize;
    private final Map<Integer, List<byte[]>> objects = new HashMap<>();

    public GCDemo() {
        blockSize = DEFAULT_BLOCK_SIZE;
    }

    public static void main(final String[] args) {
        new GCDemo().run();
    }

    private void run() {
        final Scanner in = new Scanner(System.in);
        while (true) {
            final int totalSize = objects.entrySet().stream().mapToInt(e -> e.getKey() * e.getValue().size()).sum();
            final int totalCount = objects.values().stream().mapToInt(List::size).sum();
            System.out.format(
                    "%n%nI have %d objects in use (%d MB): %s%n" +
                    "c[reate] [count=%d] [size=%d] \t- create objects%n" +
                    "r[emove] [count=%d] [size=%d] \t- remove objects%n" +
                    "s[et] [size=%d]               \t- set default block size%n" +
                    "g[c]                          \t- call System.gc()%n" +
                    "q[uit]                        \t- quit%n",
                    totalCount, totalSize,
                    objects.entrySet().stream()
                            .map(e -> e.getKey() + "x" + e.getValue().size()).collect(Collectors.joining(", ")),
                    DEFAULT_COUNT, blockSize,
                    DEFAULT_COUNT, blockSize,
                    DEFAULT_BLOCK_SIZE
            );
            if (!in.hasNextLine()) {
                break;
            }
            final Scanner line = new Scanner(in.nextLine().strip());
            if (!line.hasNext()) {
                continue;
            }

            final String command = line.next();
            if (command.startsWith("c")) {
                createObjects(optInt(line, DEFAULT_COUNT), optInt(line, blockSize));
            } else if (command.startsWith("r")) {
                removeObjects(optInt(line, DEFAULT_COUNT), optInt(line, blockSize));
            } else if (command.startsWith("s")) {
                blockSize = optInt(line, 5);
            } else if (command.startsWith("g")) {
                System.gc();
            } else if (command.startsWith("q")) {
                break;
            } else {
                System.out.format("Unknown command '%s'", command);
            }
        }

        System.out.println("Bye!");
    }

    private static int optInt(final Scanner line, final int def) {
        return line.hasNextInt() ? line.nextInt() : def;
    }

    private void createObjects(final int count, final int size) {
        try {
            System.out.format("Creating objects %d object of size %dMB%n", count, size);
            final List<byte[]> list = objects.computeIfAbsent(size, k -> new ArrayList<>());
            for (int i = 0; i < count; i++) {
                list.add(new byte[size * BYTES_IN_MB]);
            }
        } catch (final OutOfMemoryError e) {
            e.printStackTrace();
        }
    }

    private void removeObjects(final int count, final int size) {
        final List<byte[]> list = objects.get(size);
        if (list == null) {
            return;
        }

        System.out.format("Removing objects %d object of size %dMB", count, size);
        final int len = list.size();
        list.subList(Math.max(len - count, 0), len).clear();
    }
}
