package net;

import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Simple HTTP server example.
 *
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
public final class HelloWebServer {
    private static final String FORM_TEMPLATE = "" +
            "<form method='{method}'>" +
            "  <label>{method}:</label>" +
            "  <input name='hello' value='{name}'>" +
            "  <input type='submit' value='Submit'>" +
            "</form>";

    private static final String HTML_TEMPLATE = "" +
            "<html>" +
            "  <head>" +
            "    <meta charset='UTF-8'>" +
            "    <style>label {min-width: 3em; display: inline-block;}</style>" +
            "  </head>" +
            "  <body>" +
            "    <h1>Hello, {name}!</h1>" +
            "    {get}" +
            "    {post}" +
            "  </body>" +
            "</html>";

    // Utility class
    private HelloWebServer() {}

    public static void main(final String[] args) throws IOException {
        start(args.length > 0 ? Integer.parseInt(args[0]) : 8088);
    }

    private static void start(final int port) throws IOException {
        final HttpHandler helloHandler = exchange -> {
            final Map<String, List<String>> params = getParams(exchange);
            final List<String> names = params.getOrDefault("hello", List.of());
            final String name = names.isEmpty() ? "" : names.get(0);
            final String escapedName = escapeHTML(name);

            final String responseText = template(HTML_TEMPLATE, Map.of(
                    "name", escapedName,
                    "post", form("POST", escapedName),
                    "get", form("GET", escapedName))
            );
            final byte[] response = responseText.getBytes(StandardCharsets.UTF_8);
            exchange.sendResponseHeaders(HttpURLConnection.HTTP_OK, response.length);
            exchange.getResponseHeaders().set("Content-Type", "text/html; charset=UTF-8");
            try (final OutputStream os = exchange.getResponseBody()) {
                os.write(response);
            }
        };

        final HttpServer server = HttpServer.create(new InetSocketAddress(port), 0);
        server.createContext("/hello", helloHandler);
        server.setExecutor(null);
        server.start();
    }

    private static String form(final String method, final String name) {
        return template(FORM_TEMPLATE, Map.of("method", method, "name", name));
    }

    private static String template(String template, final Map<String, String> params) {
        for (final Map.Entry<String, String> entry : params.entrySet()) {
            template = template.replace("{" + entry.getKey() + "}", entry.getValue());
        }
        return template;
    }

    private static LinkedHashMap<String, List<String>> getParams(final HttpExchange t) throws IOException {
        final String params;
        if ("GET".equals(t.getRequestMethod())) {
            params = t.getRequestURI().getQuery();
        } else {
            try (final InputStream is = t.getRequestBody()) {
                params = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8)).readLine();
            }
        }

        return splitParams(Objects.requireNonNullElse(params, ""));
    }

    /**
     * Quick-and-dirty solution due to no dependencies in examples.
     * Better use Apache <a href="https://www.javadoc.io/doc/org.apache.httpcomponents/httpclient/4.5.1/org/apache/http/client/utils/URLEncodedUtils.html">URLEncodedUtils</a>
     * or Spring <a href="https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/util/UriComponentsBuilder.html">UriComponentsBuilder</a>.
     */
    private static LinkedHashMap<String, List<String>> splitParams(final String params) {
        return Arrays.stream(params.split("&"))
                .map(param -> Arrays.stream(param.split("=", 2))
                        .map(v -> URLDecoder.decode(v, StandardCharsets.UTF_8))
                        .toArray(String[]::new))
                .collect(Collectors.groupingBy(
                        kv -> kv[0],
                        LinkedHashMap::new,
                        Collectors.mapping(kv -> kv.length == 2 ? kv[1] : null, Collectors.toList())
                ));
    }

    private static final Map<Integer, String> HTML_ESCAPES = Map.of(
            '"', "&quot;",
            '\'', "&apos;",
            '<', "&lt;",
            '>', "&gt;",
            '&', "&amp;"
    ).entrySet().stream().collect(Collectors.toMap(e -> (int) e.getKey(), Map.Entry::getValue));

    /**
     * Another quick-and-dirty solution due to no dependencies in examples.
     * Better use Apache <a href="https://commons.apache.org/proper/commons-lang/apidocs/org/apache/commons/lang3/StringEscapeUtils.html">StringEscapeUtils</a>
     * or Spring <a href="https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/util/HtmlUtils.html">HtmlUtils</a>.
     */
    private static String escapeHTML(final String value) {
        return value.chars()
                .mapToObj(c -> HTML_ESCAPES.getOrDefault(c, Character.toString(c)))
                .collect(Collectors.joining());
    }
}
